/* Copyright 2017 The OpenXLA Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "xla/hlo/testlib/filecheck.h"

#include <string>

#include "absl/log/log.h"
#include "absl/status/status.h"
#include "absl/status/statusor.h"
#include "absl/strings/string_view.h"
#include "xla/tsl/platform/env.h"
#include "xla/tsl/platform/errors.h"
#include "xla/tsl/platform/resource_loader.h"
#include "xla/tsl/platform/subprocess.h"
#include "tsl/platform/path.h"
#include "tsl/platform/platform.h"

namespace xla {

absl::StatusOr<bool> RunFileCheck(const std::string& input,
                                  absl::string_view pattern) {
  // Generate an input file for the FileCheck pattern.
  std::string pattern_path;
  auto env = tsl::Env::Default();
  if (!env->LocalTempFilename(&pattern_path)) {
    return absl::InternalError("couldn't get a pattern file name");
  }
  TF_RETURN_IF_ERROR(tsl::WriteStringToFile(env, pattern_path, pattern));
  VLOG(3) << "input: " << input;
  return RunFileCheckWithPatternFile(input, pattern_path);
}

absl::StatusOr<bool> RunFileCheckWithPatternFile(
    const std::string& input, const std::string& pattern_file) {
  // Invoke FileCheck to check whether input matches `pattern`.
  std::string binary_name = "FileCheck";
  tsl::io::AppendDotExeIfWindows(binary_name);
  std::string file_check_path = tsl::GetDataDependencyFilepath(
      tsl::kIsOpenSource
          ? tsl::io::JoinPath("external", "llvm-project", "llvm", binary_name)
          : tsl::io::JoinPath("llvm", "llvm-project", "llvm", binary_name));

  tsl::SubProcess file_check_process;
#if GOOGLE_CUDA || TENSORFLOW_USE_ROCM
  std::string file_check_prefixes;
#if GOOGLE_CUDA
  file_check_prefixes = "--check-prefixes=CHECK,CHECK-PTX";
#endif  // GOOGLE_CUDA
#if TENSORFLOW_USE_ROCM
  file_check_prefixes = "--check-prefixes=CHECK,CHECK-GCN";
#endif  // TENSORFLOW_USE_ROCM
  file_check_process.SetProgram(
      file_check_path,
      {file_check_path, "-v", "-dump-input=fail", "--dump-input-filter=all",
       file_check_prefixes, "--allow-unused-prefixes", pattern_file});
#else  // !(GOOGLE_CUDA || TENSORFLOW_USE_ROCM)
  file_check_process.SetProgram(file_check_path,
                                {file_check_path, "-v", "-dump-input=fail",
                                 "--dump-input-filter=all", pattern_file});
#endif
  file_check_process.SetChannelAction(tsl::CHAN_STDIN, tsl::ACTION_PIPE);
  file_check_process.SetChannelAction(tsl::CHAN_STDERR, tsl::ACTION_PIPE);
  if (!file_check_process.Start()) {
    return absl::InternalError("couldn't start FileCheck");
  }

  std::string standard_error;
  int exit_status = file_check_process.Communicate(
      /*stdin_input=*/&input, /*stdout_output=*/nullptr,
      /*stderr_output=*/&standard_error);

  // FileCheck returns 0 when the inputs match. If matching failed, log
  // the error message generated by FileCheck and the inputs.
  bool succeeded = (exit_status == 0);
  auto env = tsl::Env::Default();
  if (!succeeded) {
    LOG(ERROR) << "Tried to execute FileCheck at " << file_check_path;
    if (!env->FileExists(file_check_path).ok()) {
      LOG(ERROR) << "NOTE: FileCheck binary does not exist!";
    }

    // Log at ERROR level so these show up even if you don't pass --logtostderr.
    LOG(ERROR) << "FileCheck input was:\n" << input;
    LOG(ERROR) << "FileCheck stderr:\n" << standard_error;
  } else if (!standard_error.empty()) {
    LOG(INFO) << "FileCheck input was:\n" << input;
    LOG(INFO) << "FileCheck stderr:\n" << standard_error;
  }
  return succeeded;
}

}  // namespace xla
